// example using the FrSkyV8Tx library to work with a cppm (buddy box) input signal
// most older R/C transmitters have a cppm signal available internally - even those
// transmitters that don't have buddy-box connectors.
// ceptimus. November 2020

// use different IDs for different transmitters that will be switched on at the same time!
// the ID number should be in the range 0 to 32767
#define TX_ID 12345

// if you want the transmitter to auto-bind for ten seconds at power up
// remove the // from the line below
// #define AUTOBIND

// if you want a transmitter bind button that will put the transmitter into bind
// (only if the button is held while powering on the transmitter)
// then make sure the AUTOBIND is left commented, remove the // from the line below
// and change the number to use the pin you've connected to the button
// the button should be normally open, and link the specified pin to GND when pressed.
// #define BIND_PIN 9

// if you know your CC2500's fine tuning value, put it here.
// if you don't know, try 0 first, then try steps of 20 in the range -120 to +120
// ...till you get a bind to your receiver
#define FINE_TUNING 0

// if you prefer to use a trim pot on an analog input for fine tuning, remove the // from
// the line below, and edit the pin number to the one you want to use for the trim pot.
// I recommend a pot in the range 10k to 100k with the wiper pin connected to the chosen
// analog input pin and the other two pot pins to VCC and GND
// #define FINE_TUNE_POT A5

// if you want a range test button that will reduce transmitter power for range testing,
// remove the // from the line below and change the number to use the pin you've connected
// to the button.  You can optionally use the same button/pin that you're using for BIND_PIN
// because the bind function only happens at start up, and the range test function will 
// operate at all times when not binding.  Be careful if you have this enabled on a transmitter
// you're using to fly a model - if you accidentally hold the button while flying, you'll get
// reduced range, possibly causing loss of control!
// #define RANGE_TEST_PIN 9

// I recommend using pin 2 or pin 3 as the input pin for the CPPM.
// These pins are hardware capable of distinguishing between low and high transitions and have
// a higher interrupt priority, so performance should be slightly better compared to other pins.
#define CPPM_PIN 2
// If you wish to use a pin other than 2 or 3 then the Arduino attachInterrupt function won't
// work.  You can still do it using pin change interrupts, but it's slightly more complicated.
// Search the internet to find how.

// if you want data (channel values, fine tuning) to be sent to the serial monitor
// remove the // from the line below, and set the number to the baud rate you want to use
// #define SERIAL_DATA 115200

#ifdef SERIAL_DATA
  int numChannels = 0; // count the channels received as CPPM data
  int channels[8];
  uint16_t ch0min = 3000;
  uint16_t ch0max = 0;
  uint32_t nextPrintMillis = 500;
#endif

#include <FrSkyV8Tx.h> // include the library
FrSkyV8Tx frsky(TX_ID, 10, FINE_TUNING); // create an instance of the class. Chip select on pin 10.
// you don't have to name the instance frsky - you can use any name you like.

void setup() {
#ifdef AUTOBIND
  frsky.bind(); // bind for ten seconds at startup
#endif

#ifdef BIND_PIN
  pinMode(BIND_PIN, INPUT_PULLUP);
  // don't move this bind() code into loop() unless you know exactly what you're doing.
  // If the code is in loop(), and you press the bind() button while flying, then you'll
  // have no control for ten seconds - probably causing a crash!
  if (digitalRead(BIND_PIN) == LOW) { // if the specifed bind pin is linked to GND
    frsky.bind(); // bind for ten seconds at startup
  }
#endif

#ifdef RANGE_TEST_PIN
  pinMode(RANGE_TEST_PIN, INPUT_PULLUP); // pin is read and controls transmit power in loop()
#endif

  frsky.start(); // start transmitting (with bind, if selected, for the first ten seconds)

  // the normal Arduino micros() function only resolves in steps of 4 us on 16 MHz Arduinos
  // and, even worse, 8 us steps on 8 MHz models.
  // this isn't really good enough for measuring CPPM signals, so instead we can use
  // the FrSkyV8Tx micros() function which resolves to 1 us on both 8 MHz and 16 MHz Arduinos
  // note that FrSkyV8Tx micros() only works while the CC2500 is transmitting and is
  // inaccurate during any bind periods.  If you're using it to time a CPPM signal, this
  // doesn't matter because there's no point in trying to set the channel data during binding.

#ifdef SERIAL_DATA
  Serial.begin(SERIAL_DATA);
  Serial.print("\nClock speed: ");
  Serial.print(F_CPU);
  Serial.print('\n');
#endif

  // configure the cppm input pin to generate an interrupt on falling edges of the signal
  pinMode(CPPM_PIN, INPUT_PULLUP);
  attachInterrupt(digitalPinToInterrupt(CPPM_PIN), cppmFallingEdge, FALLING);
}

void loop() {
#ifdef FINE_TUNE_POT
  // read the specifed analog input, convert it to the range (-128, +127) and set the fine tuning
  frsky.setFineTune(constrain(map(analogRead(FINE_TUNE_POT), 0, 1023, -128, 127), -128, 127));
#endif

#ifdef RANGE_TEST_PIN
  if (digitalRead(RANGE_TEST_PIN) == LOW) { // if the specified pin is linked to GND
    frsky.setTransmitterPower(80); // 80 is the standard FrSky power for range testing
  } else { // the specified pin isn't linked to GND
    frsky.setTransmitterPower(255); // transmit at full power
  }
#endif

#ifdef SERIAL_DATA
  if (millis() >= nextPrintMillis) {
    Serial.print("Channels: ");
    Serial.print(numChannels);
    Serial.print(": ");
    for (int i = 0; i < numChannels; i++) {
      int c = channels[i];
      if (c < 1000) Serial.print(" "); // print leading spaces for small numbers
      if (c <  100) Serial.print(" ");
      if (c <   10) Serial.print(" ");
      Serial.print(c);
      Serial.print(' ');
    }
    Serial.print("("); // min and max values for 1st channel
    Serial.print(ch0min);
    Serial.print(':');
    Serial.print(ch0max);
    Serial.print(") Tuning: ");
  #ifdef FINE_TUNE_POT
    Serial.print(constrain(map(analogRead(FINE_TUNE_POT), 0, 1023, -128, 127), -128, 127));
  #else
    Serial.print(FINE_TUNING);
  #endif
    Serial.print("  Power: ");
  #ifdef RANGE_TEST_PIN
    if (digitalRead(RANGE_TEST_PIN) == LOW) { // if the specified pin is linked to GND
      Serial.print("80"); // 80 is the standard FrSky power for range testing
    } else { // the specified pin isn't linked to GND
      Serial.print("full");
    }
  #else
    Serial.print("full");
  #endif
    Serial.print('\n'); // end of print line
    nextPrintMillis += 500; // print twice per second
  }
#endif
}

void cppmFallingEdge() { // interrupt caused by falling edge of cppm signal
  static uint32_t previousMicros = frsky.micros(); // timestamp of the previous interrupt
  static uint8_t channel = 0; // which channel (0 to 7) the signal is for
  static uint8_t notInSync = 2; // wait for 2nd valid frame before starting transmit
  uint32_t microseconds = frsky.micros(); // get current timestamp in microseconds
  sei(); // allow other interrupts once timestamp is recorded
  uint32_t interval = microseconds - previousMicros; // elapsed time since previous edge

  if (interval > 3000) { // this is the marker sync pulse indicating start of new frame
    channel = 0;
    if (notInSync) --notInSync; // at startup, count first two sync pulses before accepting values
  } else if (!notInSync && (channel < 8)) { // this is a valid pulse for one of the 8 channels
    // the .setDeglitchedChannel() method attempts to remove any short glitches from the supplied
    // data (perhaps from noisy potentiometers or similar) but at the cost of a possible
    // one-update-frame latency (delay).  With CPPM signals that's likely 20 to 25 milliseconds.
    // you can try the .setChannel() method in place of the .setDeglitchedChannel() method below.
    // .setChannel() has no latency (delay), but depending on your particular set-up, you may 
    // see a few small glitches on your servo outputs from your receiver.
    frsky.setDeglitchedChannel(channel, interval);
#ifdef SERIAL_DATA
    channels[channel] = interval;
    if (!channel) {
      if (interval < ch0min) ch0min = interval;
      if (interval > ch0max) ch0max = interval;
    }
    if (channel >= numChannels) {
      numChannels = channel + 1;
    }
#endif
    channel++; // next edge to arrive will be for the next channel
  }
  previousMicros = microseconds; // remember timestamp ready for next interrupt
}
