// 'toy' example code for a four function (Aileron, Elevator, Throttle, Rudder) transmitter
// ceptimus. November 2020

// use different IDs for different transmitters that will be switched on at the same time!
// the ID number should be in the range 0 to 32767
#define TX_ID 12345

// if you know your CC2500's fine tuning value, put it here.
// if you don't know, try 0 first, then try steps of 20 in the range -120 to +120
// ...till you get a bind to your receiver.  There are other examples that show how to
// use a trim pot for tuning.
#define FINE_TUNING 0

#include <FrSkyV8Tx.h> // include the library
FrSkyV8Tx frsky(TX_ID, 10, FINE_TUNING); // create an instance of the class. Chip select on pin 10.
// you don't have to name the instance frsky - you can use any name you like.

// connect up your transmitter's joystick pots to the first four analogue inputs.  The wiper (centre
// wire usually) connects to the Arduino pin (say A0) and the other two pot wires connect to VCC
// and GND.

void setup() {
  frsky.bind(); // auto-bind mode for ten seconds at startup.
  frsky.start(); // start transmitting (with bind for first ten seconds)
}

void loop() {
  for (int channel = 0; channel < 4; channel++ ) { // for each of the four channels
    int potPosition = analogRead(A0 + channel); // read the position of the pot
    // the first channel (number zero) is read from A0, the second from A1, and so on...
    // potPosition will be a number from 0 to 1023 and should be about 512 at the joystick centre
    // we need to convert it to the microsecond units used by radios and servos.  A typical servo
    // is centred at about 1500 microseconds, and rotates about sixty degrees either way when the 
    // microsecond value changes by plus or minus 500 from that centre value.
    // We can use the Arduino's map function to change the potPosition number to that range
    int microseconds = map(potPosition, 100, 920, 1000, 2000);
    // so now a potPosition of 100 will become 10000 microseconds and potPosition 920 becomes 2000.
    // potPositions inbetween will proportionally be mapped between 1000 and 2000 microseconds.
    // if the potPosition happens to go outside the mapped range (100 - 920) then the microsecond
    // value will also go outside its 1000-2000 range, and we don't want that, so we can use the
    // Arduino's constrain function to keep the microseconds always within the desired range
    microseconds = constrain(microseconds, 1000, 2000);
    // and now we can send that value out to the library and  CC2500 transmitter module
    frsky.setChannel(channel, microseconds);
  }
  // note that we could have combined all the (code) lines inside the for loop body into just one long line
  // frsky.setChannel(channel, constrain(map(analogRead(A0 + channel), 100, 920, 1000, 2000), 1000, 2000));
}
