/*  FrSkyV8Tx library by ceptimus
    http://http://mode-zero.uk/viewtopic.php?f=42&t=1092
    derived from Multiprotocol Tx code by Midelic and Pascal Langer(hpnuts)
    https://github.com/pascallanger/DIY-Multiprotocol-TX-Module/edit/master/README.md

 This project is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 FrSkyV8Tx is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with FrSkyV8Tx.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "FrSkyV8Tx.h"

FrSkyV8Tx::FrSkyV8Tx(uint16_t transmitterID, uint8_t CC2500chipSelectPin, int8_t fineTuningValue) { // constructor
  if ((CC2500chipSelectPin > 10) && (CC2500chipSelectPin <= 13)) { // pins 11, 12, 13 reserved for hardware SPI so can't be used
    CC2500chipSelectPin = 10; // use pin 10 instead
  }
  commonConstructor(transmitterID, CC2500chipSelectPin, fineTuningValue);
  bitBanged = false;
  SPI.begin();
}

FrSkyV8Tx::FrSkyV8Tx(uint16_t txID, uint8_t cs, uint8_t mosi, uint8_t miso, uint8_t sck, int8_t ft) { // bit-banged SPI constructor
  commonConstructor(txID, cs, ft);
  setupOutput(mosi, &mosiPort, &mosiMask);
  setupOutput(miso, &misoPort, &misoMask); 
  pinMode(miso, INPUT); // miso is actually an input (and unused in this release - but will be needed for future D8 and D16 variants)
  setupOutput(sck, &sckPort, &sckMask);
  *sckPort &= ~sckMask;
  *mosiPort |= mosiMask;
  bitBanged = true;
}

void FrSkyV8Tx::commonConstructor(uint16_t transmitterID, uint8_t CC2500chipSelectPin, int8_t fineTuningValue) {
  // build pointer and mask for fast operation of CC2500 chip select pin
  setupOutput(CC2500chipSelectPin, &csPort, &csMask);
  deselectCC2500();
  setTransmitterID(transmitterID);
  setFineTune(fineTuningValue);
  setTransmitterPower(255);
  // low channels and high channels are sent in two 15-byte packets
  // binding uses a third 15-byte packet
  packetBuffers[0] = packetBuffers[16] = packetBuffers[32] = 14; // data length
  packetBuffers[5] = 0x0F; // first four channels packet indicator (swapped to 0x00 for phase 4)
  packetBuffers[16 + 5] = 0xF0; // last four channels packet indicator - always 0xF0
  for (uint8_t ch = 0; ch < 8; ch++) {  // default channels to mid-position (1.5 ms)
    setChannel(ch, 1500);
  }
  setChannel(2, 988); // default channel 3 (probably throttle) to 100% low.
  packetBuffers[32 + 1] = 0x03; // binding packet setup
  packetBuffers[32 + 2] = 0x01;
  packetBuffers[32 + 11] = packetBuffers[32 + 12] = packetBuffers[32 + 13] = 0; 
  bindCounter = 0;
  instance = this;
}

void FrSkyV8Tx::setupOutput(int pin, volatile uint8_t* *port, uint8_t *mask) {
  if (pin < 8) {
    *port = &PORTD;
    *mask = _BV(pin);
    DDRD |= *mask;
  } else if (pin < A0) {
    *port = &PORTB;
    *mask = _BV(pin - 8);
    DDRB |= *mask;
  } else { // default to A0 - A5 otherwise
    *port = &PORTC;
    if (pin > A5) { // A6 and A7 are analogue only, above A7 doesn't exist
      pin = A5; // use A5 for those illegal values
    }
    *mask = _BV(pin - A0);
    DDRC |= *mask;
  }
}

void FrSkyV8Tx::setTransmitterID(uint16_t transmitterID) { // for changing it after the constructor (model match)
  packetBuffers[2] = packetBuffers[16 + 2] = (uint8_t)((transmitterID >> 8) & 0x007FU); // limited to 15 bits
  packetBuffers[1] = packetBuffers[16 + 1] = (uint8_t)(transmitterID & 0x00FFU);
  packetBuffers[32 + 4] = packetBuffers[2]; // binding packet has transmitterID at offsets [3][4] 
  packetBuffers[32 + 3] = packetBuffers[1]; // instead of [1][2]
  if (transmitting) {
    stop();
    start();
  }
}

void FrSkyV8Tx::setFineTune(int8_t fineTuningValue) {
  fineTune = fineTuningValue;
}

void FrSkyV8Tx::setTransmitterPower(uint8_t power) { // 255 is full power, 80 is used when binding or range testing
  txPower = power;
}

void FrSkyV8Tx::start(void) { // start transmitting
  uint8_t addr = 0;
  uint8_t b = 0;
  uint8_t i = 0;
  
  if (transmitting) return;
  
  while ((addr != 0xFF) || (b != 0xFF)) {
    addr = pgm_read_byte_near(&conf[i][0]);
    b = pgm_read_byte_near(&conf[i][1]);
    if (addr == 0x0C) { // FSCTRL0
      oldFineTune = b = fineTune;
    } else if (addr == 0x3E) { // PATABLE
      oldTxPower = b = txPower;
    }
    CC2500writeReg(addr, b);
    i++;
  }
  CC2500strobe(0x36); // SIDLE - set idle state  
  seed = 1;
  phase = 0;
  crcID = 0xD6; // compute crc on the transmitterID
  for(i = 0; i < 2; i++) {
    crcID ^= packetBuffers[2 - i];
    for (uint8_t j = 0; j < 8; j++) {
      if (crcID & 0x01) {
        crcID = (crcID >> 1) ^ 0x83;
      } else {
        crcID >>= 1;
      }
    }
  } 
  noInterrupts();
  // use timer/counter 2 to provide microsecond resolution micros
  TCCR2A = 0x00;
  TCCR2B = 0x02; // /8 clock prescale
  TIMSK2 |= 0x01;
  // use timer/counter 1 to generate timer events every 9006us in normal mode
  // or every 53456us in bind mode
  TCCR1A = 0x00;
  TCCR1B = 0x00;
  TCNT1 = 0x0000;
  OCR1A = bindCounter ? (F_CPU <= 8000000 ? 6682 - 1 : 2 * 6682 - 1) : (F_CPU <= 8000000 ? 9006 - 1 : 2 * 9006 - 1);
  TCCR1B = bindCounter ? 0x0B : 0x0A; // start in CTC mode with prescale /8 for normal use or prescale /64 when binding
  TIMSK1 = 0x02; // enable compare match interrupt
  interrupts();
  transmitting = true;
}

void FrSkyV8Tx::stop(void) { // stop transmitting
  noInterrupts();
  TCCR1A = 0x00;
  TCCR1B = 0x00;
  interrupts();
  delay(10);
  CC2500strobe(0x36); // SIDLE - set idle state
  bindCounter = 0; 
  transmitting = false;
}

void FrSkyV8Tx::bind(void) { // put transmitter into bind mode for the next ten seconds (if/when it's transmitting)
  if (bindCounter) { // bind already requested or in progress
    return;
  }
  // packet[5] is the binding index and cycles 0 5 10 15 20 25 30 35 40 45 0 5 ... while binding
  // packet[6] to packet[10] are derived from packet[5]
  packetBuffers[32 + 5] = 0;
  if (transmitting) {
    stop();
    bindCounter = 200;
    start();
  }
  bindCounter = 200;  
}

void FrSkyV8Tx::setChannel(uint8_t channelNo, int16_t us) { // channelNo 0 to 7, pulse width to servo in microseconds
  if (channelNo >= 8) {
    return;
  }
  if (us < 732) { // standard travel is 1500 +/- 512us
    us = 732; // limit to absolute maximum 150% of standard travel
  } else if (us > 2268) {
    us = 2268;
  }
  us += us >> 1; // V8 protocol uses 1.5 counts per microsecond
  // uint8_t low8 = (uint8_t)(us & 0x00FF); 
  // uint8_t high8 = (uint8_t)((us >> 8) & 0x00FF);
  uint8_t *p = packetBuffers + (channelNo < 4 ? 6 : 16 + 6) + ((channelNo & 0x03) << 1); // pointer to buffer low byte
  uint16_t *p16 = (uint16_t *)p; // pointer to update two bytes more atomically
  *p16 = us;
}

// deglitcher - useful for CPPM sampled signals, especially on 8MHz Arduinos
// remember the previous two values in odd[] and even[]
// choose the two closest values from the current, previous, and one before that
// use the latest of those two closest ones to actually update the channel
// result is that bad, single-frame glitches are ignored
// but a one-update interval delay is introduced when a genuinely steady signal begins to change
// normal CPPM frame rates are up to 25ms so that is the kind of latency involved
void FrSkyV8Tx::setDeglitchedChannel(uint8_t channelNo, int16_t us) { // channelNo 0 to 7, pulse width to servo in microseconds
  if (channelNo >= 8) {
    return;
  }
  int d1 = odd[channelNo] - even[channelNo];
  int d2 = odd[channelNo] - us;
  int d3 = even[channelNo] - us;
  if (d1 < 0) d1 = -d1;
  if (d2 < 0) d2 = -d2;
  if (d3 < 0) d3 = -d3;
  int16_t best; // the best choice to output
  if ((d1 < d2) && (d1 < d3)) { // odd and even are the closest
    best = toggle[channelNo] ? odd[channelNo] : even[channelNo]; // use the most recent of those two
  } else { 
    best = us; // use the current reading
  }
  setChannel(channelNo, best);
  if (toggle[channelNo]) {
    odd[channelNo] = us;
    toggle[channelNo] = 0;
  } else {
    even[channelNo] = us;
    toggle[channelNo] = 1;
  }
}

void FrSkyV8Tx::selectCC2500(void) { // activate CC2500 chip select pin
  *csPort &= ~csMask;
}

void FrSkyV8Tx::deselectCC2500(void) { // deactivate CC2500 chip select pin
  *csPort |= csMask;
}

void FrSkyV8Tx::CC2500strobe(uint8_t state) {
  selectCC2500();
  SPIwrite(state);
  deselectCC2500();
}

extern "C" {
  void bitBangSPIout(volatile uint8_t* mosiPort, uint8_t mosiMask, volatile uint8_t* sckPort, uint8_t sckMask, uint8_t b);
}
void FrSkyV8Tx::SPIwrite(uint8_t b) {
  if (bitBanged) {
    bitBangSPIout(mosiPort, mosiMask, sckPort, sckMask, b);
/*
    for (uint8_t mask = 0x80; mask; mask >>= 1) { // MSBFIRST
      if (b & mask) {
        *mosiPort |= mosiMask;
      } else {
        *mosiPort &= ~mosiMask;
      }
      *sckPort |= sckMask;
      *sckPort &= ~sckMask;
    }
    *mosiPort |= mosiMask;
*/   
  } else { // hardware SPI
    SPI.beginTransaction(SPISettings(6500000, MSBFIRST, SPI_MODE0));
    SPI.transfer(b);
    SPI.endTransaction();
  }
}

void FrSkyV8Tx::CC2500writeReg(uint8_t address, uint8_t b)
{
  selectCC2500();
  SPIwrite(address); 
  SPIwrite(b);
  deselectCC2500();
}

void FrSkyV8Tx::CC2500writeRegisterMulti(uint8_t address, uint8_t *b, uint8_t len) {
  selectCC2500();
  SPIwrite(0x40 | address); // WRITE_BURST
  for (uint8_t i = 0; i < len; i++) {
    SPIwrite(*b++);
  }
  deselectCC2500();
}

void FrSkyV8Tx::CC2500writeData(uint8_t *b, uint8_t len) {
  CC2500strobe(0x3B); // SFTX - flush the TX FIFO buffer
  CC2500writeRegisterMulti(0x3F, b, len); // TX FIFO
  CC2500strobe(0x35); // STX - Enable Tx
}

uint8_t FrSkyV8Tx::crc(uint8_t b, uint8_t *d, uint8_t len) {
  for (uint8_t i = 0; i < len; i++)
  {
    b ^= *d++;
    for (uint8_t j = 0; j < 8; j++) {
      if (b & 0x80) {
        b = (b << 1) ^ 0x07;
      } else {
        b <<= 1;
      }
    }
  }
  return b;
}

volatile uint32_t timer2cycles = 0UL;
uint32_t FrSkyV8Tx::micros(void) {
  uint8_t sreg = SREG;
  cli();
  uint32_t u = timer2cycles;
  uint8_t t = TCNT2;
  if (TIFR2 & 0x01) {
    t = TCNT2;
    timer2cycles = ++u;
    TIFR2 |= 0x01;
  }
  SREG = sreg;
#if F_CPU <= 8000000
  return (u << 8) | t;
#else
  return (u << 7) | (t >> 1);
#endif
}

ISR(TIMER2_OVF_vect) {
  ++timer2cycles;
}

ISR(TIMER1_COMPA_vect) {
  sei(); // allow other interrupts while servicing this timer interrupt
  FrSkyV8Tx::staticTimerEvent();
}
FrSkyV8Tx* FrSkyV8Tx::instance;
void FrSkyV8Tx::staticTimerEvent(void) {
  instance->instanceTimerEvent();
}
void FrSkyV8Tx::instanceTimerEvent(void) {
  int packetStart = 0;
  if (bindCounter) { // if binding
    if (bindCounter == 200) { // reduce transmitter power while binding
      oldTxPower = 80; // normal power will be resumed after binding
      CC2500writeReg(0x3E, oldTxPower); // PATABLE
    }
    packetBuffers[32 + 6] = packetBuffers[32 + 5] * 5 + 6;
    packetBuffers[32 + 7] = packetBuffers[32 + 5] * 5 + 11;
    packetBuffers[32 + 8] = packetBuffers[32 + 5] * 5 + 16;
    packetBuffers[32 + 9] = packetBuffers[32 + 5] * 5 + 21;
    packetBuffers[32 + 10] = packetBuffers[32 + 5] * 5 + 26;
    CC2500strobe(0x36); // SIDLE - set idle state
    CC2500writeReg(0x0A, 0x00); // channel number 0 while binding
    packetBuffers[32 + 14] = crc(0x93, packetBuffers + 32, 14);
    CC2500writeData(packetBuffers + 32, 15);
    packetBuffers[32 + 5] += 5;
    if (packetBuffers[32 + 5] > 45) {
      packetBuffers[32 + 5] = 0;
    }
    if (--bindCounter == 0) { // finished binding - back to normal operation
      OCR1A = F_CPU <= 8000000 ? 9006 - 1 : 2 * 9006 - 1;
      TCCR1B = 0x0A; // CTC mode with prescale /8 when not binding
    }
  } else { // not binding
    seed = (uint16_t)((0xAAUL * (uint32_t)seed) % 0x7673);
    // channel = (uint8_t)((seed & 0x00FF) % 0x32);
    CC2500strobe(0x36); // SIDLE - set idle state
    if (fineTune != oldFineTune) {
      CC2500writeReg(0x0C, fineTune); // FSCTRL0
      oldFineTune = fineTune;
    }
    CC2500writeReg(0x0A, (uint8_t)(((uint8_t)((seed & 0x00FF) % 0x32)) * 5 + 6)); // channel number
    packetStart = (phase & 0x01) ? 16 : 0; // send first four channels on even phases, and last four on odd
    packetBuffers[packetStart + 3] = (uint8_t)(seed & 0x00FF);
    packetBuffers[packetStart + 4] = (uint8_t)((seed >> 8) & 0x00FF);
    if (phase == 0) {
      packetBuffers[5] = 0x0F; packetBuffers[5] = 0x0F;
    } else if (phase == 4) {
      packetBuffers[5] = 0x00;
    }
    packetBuffers[packetStart + 14] = crc(crcID, packetBuffers + packetStart, 14);
    if (++phase == 5) {
      if (txPower != oldTxPower) {
        CC2500writeReg(0x3E, txPower); // PATABLE
        oldTxPower = txPower;
      }
      phase = 0;
    }
    CC2500writeData(packetBuffers + packetStart, 15);
  }
}

const PROGMEM uint8_t FrSkyV8Tx::conf[][2] = { // address:data pairs table for configuring CC2500
  {0x02, 0x06}, // IOCFG0   
  {0x00, 0x06}, // IOCFG2
  {0x17, 0x0C}, // MCSM1
  {0x18, 0x18}, // MCSM0
  {0x06, 0xFF}, // PKTLEN
  {0x07, 0x04}, // PKTCTRL1
  {0x08, 0x05}, // PKTCTRL0
  {0x3E, 0xFF}, // PATABLE
  {0x0B, 0x08}, // FSCTRL1
  {0x0C, 0x00}, // FSCTRL0
  {0x0D, 0x5C}, // FREQ2 
  {0x0E, 0x58}, // FREQ1
  {0x0F, 0x9D}, // FREQ0
  {0x10, 0xAA}, // MDMCFG4   
  {0x11, 0x10}, // MDMCFG3
  {0x12, 0x93}, // MDMCFG2
  {0x13, 0x23}, // MDMCFG1
  {0x14, 0x7A}, // MDMCFG0
  {0x15, 0x41}, // DEVIATN       
  {0x19, 0x16}, // FOCCFG
  {0x1A, 0x6C}, // BSCFG 
  {0x1B, 0x43}, // AGCCTRL2
  {0x1C, 0x40}, // AGCCTRL1
  {0x1D, 0x91}, // AGCCTRL0
  {0x21, 0x56}, // FREND1
  {0x22, 0x10}, // FREND0
  {0x23, 0xA9}, // FSCAL3
  {0x24, 0x0A}, // FSCAL2
  {0x25, 0x00}, // FSCAL1
  {0x26, 0x11}, // FSCAL0
  {0x29, 0x59}, // FSTEST
  {0x2C, 0x88}, // TEST2
  {0x2D, 0x31}, // TEST1
  {0x2E, 0x0B}, // TEST0
  {0x03, 0x07}, // FIFOTHR
  {0x09, 0x00}, // ADDR
  {0x00, 0x2F}, // IOCFG2 - TX_EN
  {0x02, 0x6F}, // IOCFG0 - TX_EN
  {0xFF, 0xFF}  // end marker
};
