//  OpenWeatherMap client testing - in preparation for dot-matrix scroller
//  ceptimus June 2021
//  edit the file config.h to specify your WiFi password, OpenWeatherMap id, location, units to use, etc.

#include <Arduino.h>
#include <ESP8266WiFi.h>
#include <ESP8266WiFiMulti.h>
#include <ESP8266HTTPClient.h>
#include <WiFiClient.h>

#include "config.h"

extern ESP8266WiFiMulti WiFiMulti;

const char* jsearch(char* token, const char* json) { // search for "token": (between quotes and with following colon) at next deepest level in *json, return pointer to the character after the colon, or NULL if not found
  int8_t level = 0;
  char c;
  
  do {
    c = *json++;
    if (c == '{') {
      level++;
    }
    else if (c == '}') {
      level--;
      if (level < 0) {
        return NULL;   
      }
    }
    else if ((level == 1) && (c == '"')) { // found an opening quote at the search level     
      c = *json++;
      bool found = true;
      char* p = token;
      while ((c >= '_') && (c <= 'z')) { // should only be lower case letters and underscores between quotes
        if (!c) {
          return NULL;
        }
        if (c != *p++) {
          found = false;
        }
        c = *json++;
        if ((c == '"') && (*json == ':')) { // found closing quote and colon
          if (*p) { // if token longer than matched text
            found = false;
          }
          if (found) {
            return ++json;
          } else { // skip (possibly quote-enclosed) field until comma is reached
            while (c != ',') {
              if (!c) {
                return NULL;
              } else if (c == '{') {
                level++;
                break;
              }
              c = *json++;
            }
          }
        }
      }
    }
  } while (c);
 
  return NULL;
}

bool findValue(const char* json, const char* search, char* result) { // return true with value *result, or false if not found
  char field[16];
  
  *result = '\0';
  
  const char* fieldStart = search;
  while (*fieldStart) {
    uint8_t index = 0;
    char c = fieldStart[index];
    while (c && c != ',') {
      field[index++] = c;
      c = fieldStart[index];
    }
    field[index] = '\0';
    const char* p = jsearch(field, json);
    if (p) { // found field
      if (c) { // search *json at next level - advance json to point at next '{'
        while (*p != '{') {
          if (*p) {
            p++;
          } else {
            return false;
          }
        }
        json = p;
      } else { // value found - return chars up to terminating comma or close-brace
        char* q = result;
        while ((*p != ',') && (*p != '}')) {
          if (*p) {
            if (*p != '"') { // remove enclosing quotes, if present
              *q++ = *p;
            }
            p++;
          } else {
            return false;
          }
        }
        *q = '\0';
        return true;
      }
    }
    fieldStart += index;
    if (c) {
      fieldStart++;
    }
  }
  return false;
}

char result[32];
char weatherString[128]; // Results from OpenWeatherMap build here, in rough Format: Studley,GB: 21.3 C, overcast clouds, wind 8-12 mph NNW

int tenthsFromASCIIFloat(char *s) { // doesn't handle negative numbers, but should be no need with OpenWeatherMap data in standard form.
  int tenths = 0;
  while ((*s < '0') || (*s > '9')) {
    if (!*s) break;
    s++;
  }
  while ((*s >= '0')  && (*s <= '9')) { // get integer part
    tenths *= 10;
    tenths += *s - '0';
    s++;
  }
  tenths *= 10;
  if (*s == '.') { // process fractional part if present
    s++;
    if ((*s >= '0')  && (*s <= '9')) { // get tenths digit
      tenths += *s - '0';
      s++;
      if ((*s >= '5')  && (*s <= '9')) { // round up if hundreths present and >= 5
        tenths++;
      }
    }
  }
  return tenths;
}


char* tenthsToNoDecimalASCII(int tenths, char* result) {
  if (tenths < 0) {
    *result++ = '-';
    tenths = -tenths;
  }
  bool outputtingDigits = false;
  for (int tenPower = 100000; tenPower >= 10; tenPower /= 10) {
    int digit = tenths / tenPower;
    if (digit || (tenPower == 10)) {
      outputtingDigits = true;
    }
    if (outputtingDigits) {
      *result++ = digit + '0';
    }
    tenths %= tenPower;
  }
  *result = '\0';
  return result;
}

void tenthsToOneDecimalASCII(int tenths, char* result) {
  result = tenthsToNoDecimalASCII(tenths, result);
  if (tenths < 0) tenths = -tenths;
  tenths %= 10;
  *result++ = '.';
  *result++ = tenths + '0';
  *result = '\0';
}

void appendToWeatherString(const char* s) {
  char* p = weatherString;
  while (*p) p++;
  while (*s) *p++ = *s++;
  *p = '\0';
}

void getWeatherString() {  
   // wait for WiFi connection
  if ((WiFiMulti.run() == WL_CONNECTED)) {

    WiFiClient client;

    HTTPClient http;

    int tenths; // result of parsing temperatures, wind speeds, to first decimal place

    weatherString[0] = '\0'; // start new string
    appendToWeatherString("No response from openweathermap.org");

    if (http.begin(client, "http://api.openweathermap.org/data/2.5/weather?q=" LOCATION "&APPID=" OPEN_WEATHER_MAP_ID)) {  // HTTP
      int httpCode = http.GET();
      if (httpCode > 0) {
        if (httpCode == HTTP_CODE_OK || httpCode == HTTP_CODE_MOVED_PERMANENTLY) {
          String payload = http.getString();
          if (findValue(payload.c_str(), "name", result)) { // get location (town or city name)
            weatherString[0] = '\0'; // start new string
            appendToWeatherString(result);
            appendToWeatherString(",");
          } else {
            http.end();
            return;
          }
          if (findValue(payload.c_str(), "sys,country", result)) { // get country
            appendToWeatherString(result);
            appendToWeatherString(": ");
          } else {
            http.end();
            return;
          }
          if (findValue(payload.c_str(), "main,temp", result)) { // get temperature in Kelvin
            tenths = tenthsFromASCIIFloat(result);
            tenths -= 2732; // convert Kelvin to Celsius
#ifdef TEMPERATURE_FAHRENHEIT
            tenths *= 9;
            tenths += 2;
            tenths /= 5;
            tenths += 320;  // convert Celsius to Fahrenheit
#endif
            tenthsToOneDecimalASCII(tenths, result);
            appendToWeatherString(result);
#ifdef TEMPERATURE_FAHRENHEIT
            appendToWeatherString(" F. ");
#else
            appendToWeatherString(" C. ");
#endif
          } else {
            http.end();
            return;
          }
          if (findValue(payload.c_str(), "weather,description", result)) { // get description, example: overcast clouds
            if ((result[0] >= 'a') && (result[0] <= 'z')) result[0] += 'A' - 'a'; // make first letter Upper case
            appendToWeatherString(result);
          } else {
            http.end();
            return;
          }
          if (findValue(payload.c_str(), "wind,speed", result)) { // get windspeed in metres per second
            appendToWeatherString(". Wind:");
            tenths = tenthsFromASCIIFloat(result);
#ifdef WIND_SPEED_MILES_PER_HOUR
            tenths *= 56;
            tenths += 12;
            tenths /= 25; // convert mps to mph (* 2.24)
#endif
            tenthsToOneDecimalASCII(tenths, result);
            appendToWeatherString(result);
          } else {
            http.end();
            return;
          }
#ifdef DISPLAY_WIND_GUST_SPEED          
          if (findValue(payload.c_str(), "wind,gust", result)) { // get wind gust speed in metres per second
            appendToWeatherString(" - ");
            tenths = tenthsFromASCIIFloat(result);
#ifdef WIND_SPEED_MILES_PER_HOUR
            tenths *= 56;
            tenths += 12;
            tenths /= 25; // convert mps to mph (* 2.24)
#endif
            tenthsToOneDecimalASCII(tenths, result);
            appendToWeatherString(result);
          } else {
            http.end();
            return;
          }
#endif
#ifdef WIND_SPEED_MILES_PER_HOUR
            appendToWeatherString(" mph ");
#else
            appendToWeatherString(" mps ");
#endif
          if (findValue(payload.c_str(), "wind,deg", result)) { // get wind direction in degrees
            tenths = tenthsFromASCIIFloat(result);
            tenths *= 10; // now hundreths of degrees
            tenths += 1125;
            tenths %= 36000;
            tenths /= 2250; // convert to 22.5-degree sector number 0 - 15
            tenths *= 3; // wind direction abbreviations have up to 3 characters
            const char* p = "N\0\0NNENE\0ENEE\0\0ESESE\0SSES\0\0SSWSW\0WSWW\0\0WNWNW\0NNW" + tenths; // pointer to the first character of the up-to three character direction abbreviation
            for (uint8_t i = 0; i < 3; i++) {
              result[i] = *p++;
            }
            result[3] = '\0';
            appendToWeatherString(result);
          } else {
            http.end();
            return;
          }
          if (findValue(payload.c_str(), "main,pressure", result)) { // get pressure in hPa (hPa = mbar)
            tenths = tenthsFromASCIIFloat(result);
            tenthsToNoDecimalASCII(tenths, result);
            appendToWeatherString(". ");            
            appendToWeatherString(result);
            appendToWeatherString(" mbar. ");
          } else {
            http.end();
            return;
          }
          if (findValue(payload.c_str(), "main,humidity", result)) { // get humidity in %
            tenths = tenthsFromASCIIFloat(result);
            tenthsToNoDecimalASCII(tenths, result);
            appendToWeatherString(result);
            appendToWeatherString(" %RH.");
          } else {
            http.end();
            return;
          }
        }
      }
      http.end();
    }
  }
}
