#include "ceptimusMAX72__.h"
#include "font.h"

ceptimusMAX72__::ceptimusMAX72__(int CS_pin, int numChips, int chipsInRow, int (*addressMap)(int)) { // constructor
  CS = CS_pin;
  digitalWrite(CS, HIGH);
  pinMode(CS, OUTPUT);
  // SPI.setBitOrder(MSBFIRST);
  SPI.begin();
  N = numChips;
  hght = N / chipsInRow;
  pWidth = (N / hght) * 8;
  pHeight = hght * 8;
  addr = addressMap;
  bitmap = new uint8_t[8 * N];
  for (int i = 0; i < MAX_ZONES; i++) {
    zoneActive[i] = false;
  }
  clr();
  wake(0);
  cmd(0x0F, 0x00); // set test mode off
  cmd(0x0B, 0x07); // set scan limit to drive 8 bytes
  cmd(0x09, 0x00); // set decode mode to no decode (not using chip's built-in seven-segment 'font')
  setBrightness(0);
  wake(1);
  refresh();
}

void ceptimusMAX72__::refresh() {
  if (!awake) {
    return;
  }
  for (int z = 0; z < MAX_ZONES; z++) {
    if (!zoneActive[z]) {
      continue;
    }
    uint32_t ms = millis();
    if ((ms < zoneNextMS[z]) && ((zoneNextMS[z] - ms) < 0x80000000UL)) { // check for millis wrap-around
      continue;
    }
    if (++zoneTextStart[z] >= widthText(zoneString[z].c_str())) {
      zoneTextStart[z] = 0;
      zoneNextMS[z] += zoneMSstart[z];
    } else {
      zoneNextMS[z] += zoneMS[z];
    }
    text(zoneX[z], zoneY[z], zoneX[z] + zoneWidth[z], zoneTextStart[z], zoneString[z].c_str(), zoneRvs[z]); 
  }
  SPI.beginTransaction(SPISettings(5000000, MSBFIRST, SPI_MODE0));
  for (int c=0; c<8; c++) { // c is the column number within each 8x8 pixel block
    digitalWrite(CS, LOW);
    for (int i=0; i<N; i++) { // i increments for each max7219 (shift register order)
      SPI.transfer(c+1); // address column (max chip uses zero for 'no operation', and 1-8 for columns)
      SPI.transfer(bitmap[c * hght + (*addr)(i)]);
    }
    digitalWrite(CS, HIGH);
  }
  SPI.endTransaction();
}

void ceptimusMAX72__::wake(int state) {
  clip (&state, 0, 1);
  awake = state;
  if (state) {
    for (int z = 0; z < MAX_ZONES; z++) {
      if (!zoneActive[z]) {
        continue;
      }
      text(z, zoneString[z].c_str(), zoneRvs[z]);
    }
  }
  cmd(0x0C, awake); // set shutdown registers to 1 (active) or 0 (shutdown)
}

void ceptimusMAX72__::setBrightness(int b) {
  if (b < 0) {
    b = 0;
  } else if (b > 15) {
    b = 15;
  }
  cmd(0x0A, b);
}

void ceptimusMAX72__::cmd(uint8_t adr, uint8_t b) { // send command to all N chips
  SPI.beginTransaction(SPISettings(5000000, MSBFIRST, SPI_MODE0));
  digitalWrite(CS, LOW);
  for (int a = 0; a < N; a++) {
    SPI.transfer(adr);
    SPI.transfer(b);
  }
  digitalWrite(CS, HIGH);
  SPI.endTransaction();
}

void ceptimusMAX72__::clr() {
  for (int a = 0; a < 8 * N; a++) {
    bitmap[a] = 0x00;
  }
}

void ceptimusMAX72__::clip(int *n, int mn, int mx) {
  if (*n > mx) {
    *n = mx;
  } else if (*n < mn) {
    *n = mn;
  }
}

void ceptimusMAX72__::setPixel(int x, int ry, bool state) {
  int y = (pHeight - 1) - ry; 
  clip(&x, 0, pWidth);
  clip(&y, 0, pHeight);

  uint8_t a = x * hght + (y >> 3);
  uint8_t mask = 0x80 >> (y & 0x07);
  if (state) {
    bitmap[a] |= mask;
  } else {
    bitmap[a] &= ~mask;
  }
}

void ceptimusMAX72__::box(int x1, int y1, int x2, int y2, bool state) {
  if (x1 > x2) {
    int t = x1; x1 = x2; x2 = t;
  }
  if (y1 > y2) {
    int t = y1; y1 = y2; y2 = t;
  }
  clip(&x1, 0, pWidth);
  clip(&x2, 0, pWidth);
  clip(&y1, 0, pHeight);
  clip(&y2, 0, pHeight);

  for (int x = x1; x <= x2; x++) {
    setPixel(x, y1, state);
    setPixel(x, y2, state);
  }

  for (int y = y1; y <= y2; y++) {
    setPixel(x1, y, state);
    setPixel(x2, y, state);
  }
}

void ceptimusMAX72__::createZone(int id, int x1, int y, int x2, int msScroll, int msScrollStart) {
  if (id < 0 || id >= MAX_ZONES) {
    return;
  }
  clip(&x1, 0, pWidth);
  clip(&x2, x1 + 1, pWidth);
  clip(&y, 0, pHeight);
  zoneX[id] = x1;
  zoneY[id] = y;
  zoneWidth[id] = x2 - x1;
  zoneMS[id] = msScroll;
  zoneMSstart[id] = msScrollStart;
  zoneString[id] = "";  
  zoneTextStart[id] = 0;
  zoneActive[id] = false;
}

// centre text in zone, if it fits, or scroll it otherwise
void ceptimusMAX72__::text(int zoneID, const char *s, bool rvs) {
  if (zoneID < 0 || zoneID >= MAX_ZONES) {
    return;
  }
  for (int y = zoneY[zoneID]; y < zoneY[zoneID] + 8; y++) {
    for (int x = zoneX[zoneID]; x < zoneX[zoneID] + zoneWidth[zoneID]; x++) {
      setPixel(x, y, rvs);
    }
  }
  int padding = zoneWidth[zoneID] - widthText(s);
  if (padding >= 0) {
    zoneActive[zoneID] = false;
  } else {
    zoneActive[zoneID] = true;
    padding = 0;
    zoneString[zoneID] = s;
    zoneRvs[zoneID] = rvs;
    zoneTextStart[zoneID] = 0;
    zoneNextMS[zoneID] = millis() + zoneMSstart[zoneID];
  }
  text(zoneX[zoneID] + padding / 2, zoneY[zoneID], zoneX[zoneID] + zoneWidth[zoneID], 0, s, rvs);
}

// clip the first [start] pixels of the text, plus any text after x2
void ceptimusMAX72__::text(int x1, int y, int x2, int start, const char *s, bool rvs) {
  clip(&x1, 0, pWidth);
  clip(&x2, x1 + 1, pWidth);
  clip(&y, 0, pHeight);
  
  while (*s) {
    int i = 0; // index into font data
    int fc = 32; // ASCII value of character at font[index];
    int w = pgm_read_byte_near(font + i); // width of that character
    while (fc < *s) { // skip through font to find character *s
      i += (w + 1);
      w = pgm_read_byte_near(font + i);
      fc++;
    }
    i++; // i now indexes first byte of font character
    for (int j = 0; j <= w; j++) { // for each byte of font plus one spacing pixel
      if (start) {
        start--;
        continue;
      }
      uint8_t b = 0; // default b to spacing pixel
      if (j < w) {
        b = pgm_read_byte_near(font + i++);; // lookup b from font except for the final spacing pixel
      }
      if (rvs) {
        b ^= 0xFF;
      }
      uint8_t mask = 0x01;
      for (int k = 0; k < 8; k++) { // for each pixel, top to bottom
        setPixel(x1, y + k, (b & mask) != 0);
        mask <<= 1; 
      }
      if (++x1 >= x2) {
        return;
      } else if (x1 > pWidth) {
        x1 = 0;
        y += 8;
      }
    }
    s++;
  }
}

int ceptimusMAX72__::widthText(const char *s) { // width of text, in pixels, includes final 'space' pixel
  int wt = 0;
  while (*s) {
    int i = 0; // index into font data
    int fc = 32; // ASCII value of character at font[index];
    int w = pgm_read_byte_near(font + i); // width of that character
    while (fc < *s) { // skip through font to find character *s
      i += (w + 1);
      w = pgm_read_byte_near(font + i);
      fc++;
    }
    wt += (w + 1);
    s++;
  }
  return wt;
}
