/*  DotMatrixClockTemperature uses ESP8266 and 32 off Max7219 8x8 dot matrix display modules
 *  display modules arranged in an 8 wide by 4 high grid. Top rows display calendar & clock,
 *  third row displays temperature from a DS18B20 sensor.
 *  fourth row displays weather data from OpenWeatherMap
 *  The prototype used a mixture of 4x1 and 4x2 modules - but could just as easily  
 *  have used all of the same type (say 4x2 for less interconnections).  The order
 *  of the chips starting at 00 is shown  below, viewed from the front. 
 *  Note that the first 'DIN' connector from the 8266 is top right because the data 
 *  shifted eventually to chip 0, is transmitted first, but passes through all the other
 *  chip's shift registers before arriving at its eventual destination.
 *  
 *  03 07 11 15 19 23 27 31
 *  02 06 10 14 18 22 26 30
 *  01 05 09 13 17 21 25 29
 *  00 04 08 12 16 20 24 28
 *  
 *  At 3.3V with all LEDs at max brightness (15), plus the 8266 itself draws ~1.5 A
 *  At 3.3V with all LEDs at min brightness (0),  plus the 8266 itself draws ~0.5 A
 *  
 *  All LEDS off, 8266 (WeMos D1 R1) draws about 100mA
 *  The AMS1117 regulator on the WeMos is nominally 1A.  May need heatsink at full brighness?
 *  
 *  ceptimus October 2020
 *  modified June 2021 to add temperature display from local DS18B20 sensor, and weather details from OpenWeatherMap.  Web interface for memos and brightness control removed from this interim version
 */
 
#include <ESP8266WiFi.h>
#include <ESP8266WiFiMulti.h>
#include <WiFiClient.h>

#include "ceptimusMAX72__.h"
#include "config.h"

// CS pin for driving display
#define CS D8
// number of 8x8 pixel blocks (and driver chips)
#define N 32
// width of display, in chips
#define CHIPS_IN_ROW 8

#define LED 2

ESP8266WiFiMulti WiFiMulti;

extern void getWeatherString(void);
extern char weatherString[];
extern char* readDS18B20(int pin, char* buffer, char scale, int offset);

char temperature[8]; // holds result from reading temperature sensor (plus scale 'F' or 'C' in temperature[6])

// bitmap address lookup function. Depends on how display modules are wired.
// see explanation in ceptimusMAX72__.h
int addressMap(int addr) {
  return ((addr & 0x7FFC) << 3) | (addr & 0x0003);
}

ceptimusMAX72__ dm(CS, N, CHIPS_IN_ROW, addressMap);

void setup(void){
  dm.createZone(0, 0,  0, 64);
  dm.createZone(1, 1,  8, 64);
  dm.createZone(2, 0, 16, 64);
  dm.createZone(3, 0, 24, 64);
  
  dm.text(0,(char *)"WiFi...");
  dm.refresh();
  
//  WiFi.begin(WIFI_SSID, WIFI_PASSWORD);
 
  //Onboard LED port Direction output
  pinMode(LED,OUTPUT); 
  //Power on LED state off
  digitalWrite(LED,HIGH);

  // power to the temperature sensor
  digitalWrite(DS18B20_ONE_WIRE_PIN, LOW);
  pinMode(DS18B20_ONE_WIRE_PIN, OUTPUT);
  digitalWrite(DS18B20_GND_PIN, LOW);
  pinMode(DS18B20_GND_PIN, OUTPUT);
  digitalWrite(DS18B20_POWER_PIN, LOW);
  pinMode(DS18B20_POWER_PIN, OUTPUT);
  delay(1000);
  pinMode(DS18B20_ONE_WIRE_PIN, INPUT_PULLUP);
  digitalWrite(DS18B20_POWER_PIN, HIGH);

  WiFi.mode(WIFI_STA);
  WiFiMulti.addAP(WIFI_SSID, WIFI_PASSWORD);
  
  // Wait for connection
  while (WiFiMulti.run() != WL_CONNECTED) {
    delay(500);
  }

  configTime(0, 0, NTP_TIME_SERVERS);
  setenv("TZ", TIME_ZONE_SETTINGS);
  time_t now;
  
  do {
    delay(500);
    time(&now);
  } while (localtime(&now)->tm_year < 120);

  dm.setBrightness(0);

#ifdef TEMPERATURE_FAHRENHEIT
  temperature[6] = 'F';
#else
  temperature[6] = 'C';
#endif
  temperature[7] = '\0';
}

void loop(void) {  
  static int32_t msToNextWeatherUpdate = 0L;
  static int32_t msToNextTemperatureUpdate = 2000L;
  static uint32_t prevMillis = millis();
  uint32_t ms = millis();
  uint32_t elapsed = ms - prevMillis;
  prevMillis = ms;
  msToNextWeatherUpdate -= elapsed;
  if (msToNextWeatherUpdate <= 0L) {
    getWeatherString();
    dm.text(3, weatherString);
    msToNextWeatherUpdate += WEATHER_UPDATE_INTERVAL;
  }

  msToNextTemperatureUpdate -= elapsed;
  if (msToNextTemperatureUpdate <= 0L) {
    readDS18B20(DS18B20_ONE_WIRE_PIN, temperature, temperature[6], DS18B20_TEMPERATURE_CORRECTION);
    temperature[5] = ' ';
    dm.text(2, temperature);
    msToNextTemperatureUpdate += TEMPERATURE_UPDATE_INTERVAL;
  }
  
  time_t now;
  time(&now);
  char bufr[30];
  strftime(bufr, 30, "%a %d %b", localtime(&now));

  dm.text(0, bufr);

  strftime(bufr, 30, "%T %Z", localtime(&now));

#ifdef TWELVE_HOUR_CLOCK
  uint8_t hr = (bufr[0] - '0') * 10 + bufr[1] - '0';
  bufr[5] = ' '; // overwrite seconds separator with space
  bufr[6] = hr < 12 ? 'A' : 'P';
  bufr[7] = 'M'; // the M of AM or PM
  if (!hr) {
    hr = 12;
  } else if (hr > 12) {
    hr -= 12;
  }
  bufr[0] = hr < 10 ? ' ' : '1';
  bufr[1] = (hr % 10) + '0';
#endif
    
  char *c;
  for (c = bufr; *c; c++) {
    if (*c == '1') {
      *c = '\177'; // replace narrow '1's with fixed-width '1's
    }
  }

#ifdef TWELVE_HOUR_CLOCK
  if (bufr[0] == ' ') {
    dm.text(1, bufr + 1);
  } else {
    dm.text(1, bufr);
  }
#else
  dm.text(1, bufr);
#endif
  dm.refresh();
}
