#include <Arduino.h>

#include "config.h"

bool resetDS18B20(int pin) { // returns true if device signals 'present' or false otherwise
  digitalWrite(pin, LOW);
  pinMode(pin, OUTPUT);
  delayMicroseconds(481);
  pinMode(pin, INPUT_PULLUP);
  delayMicroseconds(70);
  bool result = digitalRead(pin) == LOW;
  delayMicroseconds(500);
  return result;
}

void writeDS18B20byte(uint8_t b, int pin) {
  for (uint8_t i = 0; i < 8; i++) {
    if (b & 1) { // send a '1' bit - pull low for 1us to 60us
      digitalWrite(pin, LOW);
      pinMode(pin, OUTPUT);
      delayMicroseconds(2);
      pinMode(pin, INPUT_PULLUP);
      delayMicroseconds(65);
    } else { // send a '0' bit - pull low for 60us to 120 us
      digitalWrite(pin, LOW);
      pinMode(pin, OUTPUT);
      delayMicroseconds(65);
      pinMode(pin, INPUT_PULLUP);
      delayMicroseconds(2);
    }
    b >>= 1;
  }
}

void readDS18B20(int pin, uint8_t* buffer, int numBytes) {
  while (numBytes--) {
    uint8_t b = 0;
    for (uint8_t i = 0; i < 8; i++) {
      b >>= 1;
      digitalWrite(pin, LOW);
      pinMode(pin, OUTPUT);
      delayMicroseconds(2);
      pinMode(pin, INPUT_PULLUP);
      delayMicroseconds(15);
      if (digitalRead(pin) == HIGH) {
        b |= 0x80;
      }
      delayMicroseconds(46);
    }
    *buffer++ = b;
  }
}

// return ASCII temperature with one decimal place right justified in buffer (6 chars including terminating \0). scale should be 'F' or defaults to C
// offset (if provided) is in 1/16 degrees C and allows the readings from fake DS18B20s to be offset to agree more closely with a reference thermometer
char* readDS18B20(int pin, char* buffer, char scale = 'C', int offset = 0) { // return ASCII temperature with one decimal place right justified in buffer (6 chars including terminating \0). scale should be 'F' or defaults to C
  union {
    uint8_t b[2];
    int16_t t;
  } u;

  if (resetDS18B20(pin)) {
    writeDS18B20byte(0xCC, pin); // send 'Skip Rom' command
    writeDS18B20byte(0xBE, pin); // send 'Read Scratchpad' command.
    readDS18B20(pin, u.b, 2); // read temperature bytes - gives temperature in degrees C, multiplied by 16. range -55 C to +125 C
    u.t += offset;
    
    if (scale == 'F' || scale == 'f') { // convert to fahrenheit * 16
      u.t *= 9;
      u.t += 2;
      u.t /= 5;
      u.t += 32 * 16;
    }
    bool negative = u.t < 0;
    if (negative) {
      u.t = -u.t;
    }
    u.t *= 10;
    u.t += 8;
    u.t >>= 4; // now u.t is in tenths of degrees
    int16_t degrees = u.t / 10;
    int16_t tenths = u.t - degrees * 10;
    buffer[4] = tenths + '0';
    if (degrees > 199) {
      buffer[0] = '2';
      degrees -= 200;
    } else if (degrees > 99) {
      buffer[0] = '1';
      degrees -= 100;
    } else if (negative && degrees > 9) {
      buffer[0] = '-';
    } else {
      buffer[0] = ' ';
    }
    if (degrees > 9) {
      uint16_t tens = degrees / 10;
      buffer[1] = tens + '0';
      degrees -= tens * 10;
    } else if (negative) {
      buffer[1] = '-';
    } else {
      buffer[1] = ' ';
    }
    buffer[2] = degrees + '0';
    resetDS18B20(pin);
    writeDS18B20byte(0xCC, pin); // send 'Skip Rom' command
    writeDS18B20byte(0x44, pin); // send 'Convert T' command.  Conversion takes 750ms at default (12-bit) resolution
  } else { // return ---.-\n for non-reading
    for (uint8_t i = 0; i < 5; i++) {
      buffer[i] = '-';
    }
  }
  buffer[3] = '.';
  buffer[5] = '\0';
  return buffer;
}
